
- `RemoteLogin` en `RemoteLoginCallback`: Autorisatie van gebruiker bij beschikbaarstellend PGO
- `DownloadDocumentsBundle`, `StoreDocument`: Ophalen van bundel met FHIR DocumentReference.
- `DownloadFromUrl`: Uitgesteld downloaden van gelinkte DocumentReference.content.attachment.url op achtergrond.

```cs

    /// <summary>
    /// Start remote authorization and consent flow by redirecting user to the APD's or DVA's authorization endpoint.
    /// We'll receive a code in the callback when the user returns.
    /// </summary>
    [HttpGet("api/import/login/{provider}")]
    public IActionResult RemoteLogin(string? provider)
    {
        var remotePgo = _stelselnode
            .GetZorgaanbieders()
            .Where(provider => provider.ProviderType == StelselNodeProviderType.AanbiederPersoonsdomein)
            .FirstOrDefault(adp => adp?.Name == provider);

        if (remotePgo == null)
        {
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Aanbieder {provider} niet gevonden."));
        }

        // Redirect the user to authorization endpoint, include a redirect_uri pointing back to our callback.
        // Generate a state to prevent CSRF attacks, include scope 51 for consent to document exchange, etc.
        var requestId = Guid.NewGuid().ToString("N");
        var correlationId = Guid.NewGuid().ToString("N");
        var scope = HttpUtility.UrlEncode(_config.RemoteIdentScopes); // 51
        var redirectUri = HttpUtility.UrlEncode(_config.RemoteIdentRedirectUri); // api/import/login_callback
        var state = LoginState.Create(remotePgo.Name);
        var endpoint = $"{remotePgo.AuthorizationEndpoint}?client_id={_config.RemoteIdentClientId}&redirect_uri={redirectUri}&response_type=code&scope={scope}&state={state}&MedMij-Request-ID={requestId}&X-Correlation-ID={correlationId}";
        HttpContext.Session.SetString("oauth_state", state);

        return Redirect(endpoint);
    }

```

```cs

    /// <summary>
    /// After visiting remote authorization endpoint, the user is redirected back with a code or an error.
    /// We exchange the code for a token and store it in user session to exchange documents with remote ADP.
    /// </summary>
    /// <returns></returns>
    [HttpGet("api/import/login_callback")]
    public async Task<IActionResult> RemoteLoginCallback([FromQuery] string? code, [FromQuery] string? state, [FromQuery] string? scope, [FromQuery] string? error, [FromQuery] string? error_description)
    {
        // Sanity checks. Is this the same user as the one that started the flow?
        var loginState = LoginState.Parse(state);
        var sessionState = HttpContext.Session.GetString("oauth_state");                
        if (string.IsNullOrEmpty(state) || sessionState != state)
        {
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Aanbieder stuurde een onjuiste session state mee."));
        }

        // Is there an error message in the callback?
        if (!string.IsNullOrWhiteSpace(error))
        {
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Aanbieder {loginState.ProviderName} gaf geen code mee, maar toonde deze foutmelding: {error} {error_description}"));
        }

        // Fetch the token exchange endpoint from ZAL for remote PGO in this session.
        var remotePgo = _stelselnode
            .GetZorgaanbieders()
            .Where(provider => provider.ProviderType == StelselNodeProviderType.AanbiederPersoonsdomein)
            .FirstOrDefault(adp => adp?.Name == loginState.ProviderName);

        if (remotePgo == null)
        {
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Aanbieder {loginState.ProviderName} niet gevonden."));
        }

        // Exchange the authorization_code for an access token and store that in user session.
        try
        {
            var httpClient = _httpClientFactory.CreateClient();
            var tokenResponse = await httpClient.PostAsync(remotePgo.TokenEndpoint, new FormUrlEncodedContent(
                new Dictionary<string, string>
                {
                    { "grant_type", "authorization_code" },
                    { "code", code },
                    { "redirect_uri", _config.RemoteIdentRedirectUri },
                    { "client_id", _config.RemoteIdentClientId },
                }
            ));

            if (!tokenResponse.IsSuccessStatusCode)
            {
                var tokenResponseError = await tokenResponse.Content.ReadAsStringAsync();
                throw new HttpRequestException($"HTTP {tokenResponse.StatusCode}, {tokenResponseError}");
            }
            else
            {
                var token = await tokenResponse.Content.ReadFromJsonAsync<TokenResponse>();
                if (token == null)
                {
                    throw new JsonException("No token message found in body of succesful token response.");
                }
                HttpContext.Session.SetString("access_token", token.AccessToken);
            }
        }
        catch (Exception ex)
        {
            _logger.LogError("Token request failed: {0}", ex.Message);
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Aanbieder {loginState.ProviderName} gaf een code mee, maar liet deze niet inwisselen voor een token."));
        }

        return Redirect($"{_config.BasePath}/api/import/download-documents");
    }

```

```cs

    /// <summary>
    /// Fetch a bundle of FHIR DocumentReferences for user and retrieve linked PDF/A documents.
    /// Some documents may be stored as inline data, others are referenced by content.attachment.url.
    /// </summary>
    [HttpGet("api/import/download-documents")]
    public async Task<ActionResult> DownloadDocumentsBundle()
    {
        // DEMO: Sanity check.
        var userId = User.FindFirst("sub")?.Value ?? "voorbeeldgebruiker";
        var sessionState = LoginState.Parse(HttpContext.Session.GetString("oauth_state"));
        var accessToken = HttpContext.Session.GetString("access_token");
        var remotePgo = _stelselnode
            .GetZorgaanbieders()
            .Where(provider => provider.ProviderType == StelselNodeProviderType.AanbiederPersoonsdomein)
            .FirstOrDefault(adp => adp?.Name == sessionState.ProviderName);

        if (sessionState == null || accessToken == null || remotePgo == null)
        {
            return Redirect(_frontendCancelledUrl + "?error=" + HttpUtility.UrlEncode($"Tijdens het downloaden ontbrak toegangsinformatie."));
        }

        // DEMO: Fetch bundle of FHIR DocumentReferences resources.
        var fhirClient = CreateFhirClient(remotePgo.FhirResourceEndpoint, accessToken);
        var bundle = await fhirClient.SearchAsync<DocumentReference>(new SearchParams().Where("type=http://loinc.org|34133-9"));
        foreach (var entry in bundle?.Entry ?? [])
        {
            switch (entry.Resource.TypeName)
            {
                case "Patient":
                    // In this proof-of-concept we don't process structured data for the patient as the aim is to simply fetch documents.
                    // PGO may benefit though from processing the patient record that is passed or referenced by DocumentReference.subject.
                    var patient = entry.Resource as Patient;
                    break;

                case "DocumentReference":
                    // Read the document attachment and store it for the user in our local FHIR server with remote APD as source system.
                    // For remote servers it may take a long time to generate the document. So, we download documents in the background.
                    var documentReference = entry.Resource as DocumentReference;
                    foreach (var content in documentReference?.Content ?? [])
                    {
                        if (content == null) continue;

                        // Store inline attachment.data or fetch attachment.url.
                        var contentType = content.Attachment?.ContentType ?? throw new ArgumentNullException("ContentType");
                        var hasInlineData = content.Attachment?.Data != null && content.Attachment?.Data != Array.Empty<byte>();
                        if (hasInlineData)
                        {
                            await StoreDocument(userId, remotePgo.Name, content.Attachment);
                        }
                        else
                        {
                            // This is a simple example to push work to background while we redirect the user back to frontend.
                            // Helpful as concept, more appropriate to use an IHostedService, BackgroundService, HangFire, etc.
                            var contentUrl = content.Attachment?.Url ?? "";
                            var contentUrlFullyQualified = Url.IsLocalUrl(contentUrl) ? $"{fhirClient.Endpoint.Scheme}://{fhirClient.Endpoint.Host}:{fhirClient.Endpoint.Port}{contentUrl}" : contentUrl;
                            _ = Task.Run(async () =>
                            {
                                try
                                {
                                    _notificationService.UpdateStatus(userId, isDownloading: true);
                                    var data = await DownloadFromUrl(contentUrlFullyQualified);
                                    await StoreDocument(userId, remotePgo.Name, content.Attachment, data);
                                }
                                catch (Exception ex)
                                {
                                    _logger.LogError("Download of document failed: {0}", ex.Message);
                                }
                                finally
                                {
                                    _notificationService.UpdateStatus(userId, isDownloading: false);
                                }
                            });
                        }
                    }
                    break;

                default:
                    break;
            }
        }

        return Redirect(_frontendSuccessUrl);
    }

```

```cs

    /// <summary>
    /// Store a document in our local FHIR server. We create a new DocumentReference for our own administration
    /// and copy some of the metadata from the original referenced attachment. Available inline data is copied too.
    /// </summary>
    private async Task StoreDocument(string userPatientId, string provider, Attachment attachment, byte[]? dataImportedFromAttachmentUrl = null)
    {
        var fhirClient = CreateFhirClient(_config.FhirEndpoint);

        var resource = new DocumentReference
        {
            Status = DocumentReferenceStatus.Current,
            Identifier = new List<Identifier> { new Identifier("https://pgo-doel/bronsystemen", provider) },
            Type = new CodeableConcept("http://loinc.org", "34133-9", "Samenvatting medische gegevens uit ander bronsysteeem"),
            Subject = new ResourceReference($"Patient/{userPatientId}"),
            Indexed = DateTimeOffset.UtcNow,
            Content = new()
            {
                new()
                {
                    Attachment = new()
                    {
                        ContentType = attachment.ContentType ?? throw new ArgumentNullException("ContentType"),
                        Data = dataImportedFromAttachmentUrl ?? attachment.Data,
                        Title = attachment.Title ?? $"Document van {provider}",
                    }
                }
            }
        };

        await fhirClient.CreateAsync(resource);
    }
```

```cs

    /// <summary>
    /// Download a document attachment from url. The remote server may return a 202 Accepted instead of 200 OK
    /// if generating the document takes time. We poll the status endpoint until the document is ready.
    /// </summary>
    private async Task<byte[]> DownloadFromUrl(string url)
    {
        if (Debugger.IsAttached)
        {
            // Quick override for local testing.
            url = "https://localhost:5001/api/resources/genereer_dossier_pdf";
        }

        var httpclient = _httpClientFactory.CreateClient("WebClient");
        var originalUri = new Uri(url);
        var timeoutAfter = DateTime.UtcNow.AddSeconds(5 * 60);

        while (true)
        {
            if (DateTime.UtcNow > timeoutAfter)
            {
                throw new TimeoutException($"Timeout while waiting for document at {url}");
            }

            HttpResponseMessage response = await httpclient.GetAsync(url);
            switch (response.StatusCode)
            {
                // HTTP 200 OK at api/resources/genereer_dossier_pdf
                case HttpStatusCode.OK:
                    return await response.Content.ReadAsByteArrayAsync();

                // HTTP 202 Accepted, Location: /api/resources/status/123
                // HTTP 200 Accepted, RetryAfter: 2
                case HttpStatusCode.Accepted:
                case HttpStatusCode.TooManyRequests:
                case HttpStatusCode.ServiceUnavailable:
                    if (response.Headers.Location != null)
                    {
                        if (Url.IsLocalUrl(response.Headers.Location.ToString()))
                        {
                            url = new Uri(originalUri, response.Headers.Location).ToString();
                        }
                        else
                        {
                            url = response.Headers.Location.ToString();
                        }
                    }
                    int retryAfter = response.Headers.RetryAfter == null ? 5 : response.Headers.RetryAfter.Delta?.Seconds ?? 5;
                    await Task.Delay(TimeSpan.FromSeconds(retryAfter));
                    break;

                default:
                    throw new HttpRequestException($"Unexpected response in {nameof(DownloadFromUrl)}: {response.StatusCode}");
            }
        }
    }
}

```